import numpy as np
from scipy.integrate import quad
import os

# --- CONSTANTS ---
C_LIGHT = 299792.458
MPC_TO_KM = 3.08567758e19
SEC_TO_GYR = 1.0 / (365.25 * 24 * 3600 * 1e9)
SEC_TO_MYR = 1.0 / (365.25 * 24 * 3600 * 1e6)

# --- PARAMETERS ---
K_EXP = 4224.0 
K_SIG = 3232.0
ETA_SIG = 1.12
Z_START = 20.0
Z_END = 14.0
CONV_MULTS = [10.0, 100.0, 1000.0]

def v0_exp(r_mpc):
    return C_LIGHT * (1.0 - np.exp(-r_mpc / K_EXP))

def v0_sigmoid(r_mpc):
    term = (r_mpc / K_SIG) ** ETA_SIG
    return C_LIGHT * (term / (1.0 + term))

def n_from_v(v):
    beta = v / C_LIGHT
    return np.sqrt((1.0 + beta**2) / (1.0 - beta**2))

def gamma_from_v(v):
    beta2 = (v / C_LIGHT) ** 2
    return 1.0 / np.sqrt(1.0 - beta2)

def r_from_z(z, key):
    if key == 'exp': return K_EXP * np.log(1.0 + z)
    elif key == 'sigmoid': return K_SIG * (z ** (1.0/ETA_SIG))
    return 0.0

def integrand_proper_time(r, v_func):
    v = v_func(r)
    n = n_from_v(v)
    g = gamma_from_v(v)
    return n / (C_LIGHT * g)

def integrand_coord_time(r, v_func):
    v = v_func(r)
    n = n_from_v(v)
    return n / C_LIGHT

def integrate_quad(func, r0, r1):
    res, err = quad(func, r0, r1)
    return res * MPC_TO_KM

def main():
    os.makedirs('produced', exist_ok=True)
    out_file = 'produced/appendix_D_time_dilation.txt'
    
    report = []
    def log(s):
        print(s)
        report.append(s)

    log('--- REFRACTIVE COSMOLOGY TIME DILATION (Appendix D) ---')
    
    models = [('Exponential', 'exp', v0_exp), ('Sigmoidal (Hill)', 'sigmoid', v0_sigmoid)]

    for name, key, func in models:
        r_start = r_from_z(Z_START, key)
        r_end = r_from_z(Z_END, key)
        
        sec_window = integrate_quad(lambda r: integrand_proper_time(r, func), r_end, r_start)
        myr_window = sec_window * SEC_TO_MYR
        
        sec_coord = integrate_quad(lambda r: integrand_coord_time(r, func), 0, r_end)
        gyr_coord = sec_coord * SEC_TO_GYR
        
        log(f'\n[Model: {name}]')
        log(f' r(z=20): {r_start:.1f} Mpc')
        log(f' r(z=14): {r_end:.1f} Mpc')
        log(f' Coordinate time to z=14: {gyr_coord:.2f} Gyr')
        log(f' Proper evolutionary window z=20 -> 14: {myr_window:.1f} Myr')
        
        for mult in CONV_MULTS:
            r_far = r_end + mult * (K_EXP if key=='exp' else K_SIG)
            sec_tail = integrate_quad(lambda r: integrand_proper_time(r, func), r_end, r_far)
            gyr_tail = sec_tail * SEC_TO_GYR
            log(f'  Tail +{int(mult)}K: +{gyr_tail:.2f} Gyr')

    with open(out_file, 'w') as f:
        f.write('\n'.join(report))
    print(f"\nReport saved to {out_file}")

if __name__ == '__main__':
    main()